bl_info = {
    "name": "Custom Splash 自定义启动画面",
    "author": "deathblood",
    "version": (1, 0, 0),
    "blender": (4, 4, 0),
    "location": "Help Menu",
    "description": "在系统内设置 Blender 启动画面,适用于4.4后所有版本，位于帮助菜单中",
    "category": "Interface",
}

import bpy
import os
import sys
import winreg
import ctypes
import subprocess
from bpy.types import Operator, Menu


def is_admin():
    try:
        return ctypes.windll.shell32.IsUserAnAdmin()
    except:
        return False


def set_system_env_var(name, value):
    try:
        key = winreg.OpenKey(
            winreg.HKEY_LOCAL_MACHINE,
            r"SYSTEM\CurrentControlSet\Control\Session Manager\Environment",
            0,
            winreg.KEY_SET_VALUE
        )
        winreg.SetValueEx(key, name, 0, winreg.REG_SZ, value)
        winreg.CloseKey(key)

        # 通知系统刷新环境变量
        HWND_BROADCAST = 0xFFFF
        WM_SETTINGCHANGE = 0x1A
        SMTO_ABORTIFHUNG = 0x0002
        result = ctypes.c_long()
        ctypes.windll.user32.SendMessageTimeoutW(
            HWND_BROADCAST,
            WM_SETTINGCHANGE,
            0,
            "Environment",
            SMTO_ABORTIFHUNG,
            5000,
            ctypes.byref(result)
        )
        return True

    except PermissionError:
        return False
    except Exception as e:
        print(f"设置系统变量失败: {e}")
        return False


def remove_system_env_var(name):
    try:
        key = winreg.OpenKey(
            winreg.HKEY_LOCAL_MACHINE,
            r"SYSTEM\CurrentControlSet\Control\Session Manager\Environment",
            0,
            winreg.KEY_SET_VALUE
        )
        winreg.DeleteValue(key, name)
        winreg.CloseKey(key)

        # 通知系统刷新
        HWND_BROADCAST = 0xFFFF
        WM_SETTINGCHANGE = 0x1A
        SMTO_ABORTIFHUNG = 0x0002
        result = ctypes.c_long()
        ctypes.windll.user32.SendMessageTimeoutW(
            HWND_BROADCAST,
            WM_SETTINGCHANGE,
            0,
            "Environment",
            SMTO_ABORTIFHUNG,
            5000,
            ctypes.byref(result)
        )
        return True

    except FileNotFoundError:
        return False
    except PermissionError:
        return False
    except Exception as e:
        print(f"删除系统变量失败: {e}")
        return False


def get_system_env_var(name):
    """获取系统级环境变量"""
    try:
        key = winreg.OpenKey(
            winreg.HKEY_LOCAL_MACHINE,
            r"SYSTEM\CurrentControlSet\Control\Session Manager\Environment",
            0,
            winreg.KEY_READ
        )
        value, _ = winreg.QueryValueEx(key, name)
        winreg.CloseKey(key)
        return value
    except FileNotFoundError:
        return None
    except Exception as e:
        print(f"读取系统变量失败: {e}")
        return None


def refresh_environment():
    """强制刷新系统环境变量（需要管理员权限）"""
    try:
        # 使用广播消息
        HWND_BROADCAST = 0xFFFF
        WM_SETTINGCHANGE = 0x1A
        SMTO_ABORTIFHUNG = 0x0002
        result = ctypes.c_long()
        ctypes.windll.user32.SendMessageTimeoutW(
            HWND_BROADCAST,
            WM_SETTINGCHANGE,
            0,
            "Environment",
            SMTO_ABORTIFHUNG,
            5000,
            ctypes.byref(result)
        )
        return True
    except Exception as e:
        print(f"刷新环境变量失败: {e}")
        return False


class CUSTOM_OT_set_splash_system(Operator):
    bl_idname = "custom.set_splash_system"
    bl_label = "设置系统级自定义启动画面"
    bl_description = "选择图片文件进行设置（需管理员权限）"
    bl_options = {'REGISTER'}

    filepath: bpy.props.StringProperty(subtype='FILE_PATH')
    filter_glob: bpy.props.StringProperty(default="*.png;*.jpg;*.jpeg", options={'HIDDEN'})

    @classmethod
    def poll(cls, context):
        """只有在管理员模式下才允许操作"""
        return is_admin()

    def execute(self, context):
        if not self.filepath or not os.path.exists(self.filepath):
            self.report({'ERROR'}, "无效文件路径")
            return {'CANCELLED'}

        abs_path = os.path.abspath(self.filepath)

        ok = set_system_env_var("BLENDER_CUSTOM_SPLASH", abs_path)
        if ok:
            self.report({'INFO'}, f"✅ 已设置系统级启动画面: {abs_path}")
            
            # 显示重启提示
            def restart_callback(self, context):
                layout = self.layout
                layout.label(text="启动画面已设置！")
                layout.label(text="需要重启Blender才能生效")
                layout.separator()
                row = layout.row()
                row.operator("wm.quit_blender", text="立即重启")
            
            context.window_manager.popup_menu(restart_callback, title="设置成功", icon='INFO')
            
            return {'FINISHED'}
        else:
            self.report({'ERROR'}, "设置失败，请确保以管理员权限运行")
            return {'CANCELLED'}

    def invoke(self, context, event):
        if not is_admin():
            self.report({'WARNING'}, "需要管理员权限，请先以管理员身份运行Blender")
            return {'CANCELLED'}
        
        context.window_manager.fileselect_add(self)
        return {'RUNNING_MODAL'}


class CUSTOM_OT_remove_splash_system(Operator):
    """移除系统级启动画面"""
    bl_idname = "custom.remove_splash_system"
    bl_label = "移除启动画面"
    bl_description = "移除启动画面设置（需管理员权限）"
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        """只有在管理员模式下才允许操作"""
        return is_admin()

    def execute(self, context):
        ok = remove_system_env_var("BLENDER_CUSTOM_SPLASH")
        if ok:
            self.report({'INFO'}, "🗑️ 已移除系统级启动画面设置")
            
            # 显示重启提示
            def restart_callback(self, context):
                layout = self.layout
                layout.label(text="启动画面已移除！")
                layout.label(text="需要重启Blender才能生效")
                layout.separator()
                row = layout.row()
                row.operator("wm.quit_blender", text="立即重启")
            
            context.window_manager.popup_menu(restart_callback, title="移除成功", icon='INFO')
            
        else:
            self.report({'ERROR'}, "移除失败，请确保以管理员权限运行")
        return {'FINISHED'}


class CUSTOM_OT_run_as_admin(Operator):
    """重新以管理员身份运行 Blender"""
    bl_idname = "custom.run_as_admin"
    bl_label = "以管理员身份运行Blender"
    bl_description = "需管理员权限才能设置系统变量"

    def execute(self, context):
        try:
            blender_path = bpy.app.binary_path
            current_file = bpy.data.filepath if bpy.data.filepath else ""
            
            if current_file:
                cmd = f'Start-Process -FilePath "{blender_path}" -ArgumentList "{current_file}" -Verb runAs'
            else:
                # 如果没有打开的文件，直接启动
                cmd = f'Start-Process -FilePath "{blender_path}" -Verb runAs'
            
            subprocess.Popen(["powershell", "-Command", cmd], shell=True)
            
            # 延迟退出当前实例
            bpy.app.timers.register(lambda: bpy.ops.wm.quit_blender(), first_interval=1.0)
            
        except Exception as e:
            self.report({'ERROR'}, f"启动失败: {e}")
        return {'FINISHED'}


class CUSTOM_OT_refresh_environment(Operator):
    """刷新系统环境变量"""
    bl_idname = "custom.refresh_environment"
    bl_label = "刷新系统环境变量"
    bl_description = "强制刷新系统环境变量使设置立即生效（需管理员权限）"
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        """只有在管理员模式下才允许操作"""
        return is_admin()

    def execute(self, context):
        # 检查当前是否已有设置
        current_splash = get_system_env_var("BLENDER_CUSTOM_SPLASH")
        if not current_splash:
            self.report({'WARNING'}, "未找到启动画面设置")
            return {'CANCELLED'}
        
        # 尝试刷新环境变量
        ok = refresh_environment()
        if ok:
            self.report({'INFO'}, "✅ 环境变量刷新成功")
            
            # 显示当前设置信息
            def show_info_callback(self, context):
                layout = self.layout
                layout.label(text=f"当前启动画面:", icon='IMAGE_DATA')
                layout.label(text=os.path.basename(current_splash))
                layout.separator()
                layout.label(text="需要重启Blender才能看到效果")
            
            context.window_manager.popup_menu(show_info_callback, title="刷新成功", icon='INFO')
        else:
            self.report({'ERROR'}, "刷新失败")
        
        return {'FINISHED'}


class CUSTOM_OT_show_current_splash(Operator):
    """显示当前启动画面设置"""
    bl_idname = "custom.show_current_splash"
    bl_label = "当前设置状态"
    bl_description = "显示当前系统级启动画面设置状态"
    bl_options = {'REGISTER'}

    def execute(self, context):
        current_splash = get_system_env_var("BLENDER_CUSTOM_SPLASH")
        is_admin_mode = is_admin()
        
        def status_callback(self, context):
            layout = self.layout
            
            if current_splash and os.path.exists(current_splash):
                layout.label(text="✅ 启动画面已设置", icon='CHECKMARK')
                layout.separator()
                layout.label(text=f"图片路径:", icon='IMAGE_DATA')
                layout.label(text=current_splash)
                layout.separator()
                layout.label(text=f"文件存在: {os.path.exists(current_splash)}")
                if is_admin_mode:
                    layout.operator("custom.refresh_environment", text="刷新环境变量", icon='FILE_REFRESH')
            else:
                layout.label(text="❌ 未设置启动画面", icon='ERROR')
                if not is_admin_mode:
                    layout.separator()
                    layout.label(text="请先以管理员身份运行Blender")
                    layout.label(text="然后设置启动画面图片")
        
        context.window_manager.popup_menu(status_callback, title="启动画面状态", icon='PREFERENCES')
        return {'FINISHED'}


class CUSTOM_MT_splash_submenu(Menu):
    """自定义启动画面子菜单"""
    bl_idname = "CUSTOM_MT_splash_submenu"
    bl_label = "自定义启动画面"

    def draw(self, context):
        layout = self.layout
        is_admin_mode = is_admin()
        
        current_splash = get_system_env_var("BLENDER_CUSTOM_SPLASH")
        if current_splash and os.path.exists(current_splash):
            layout.operator("custom.show_current_splash", text="✅ 启动画面已设置")
        else:
            layout.operator("custom.show_current_splash", text="❌ 启动画面未设置")
        
        layout.separator()
        layout.operator("custom.run_as_admin", icon='LOCKED')
        layout.separator()
        
        if is_admin_mode:
            layout.operator("custom.set_splash_system",  text="1.设置启动画面",icon='FILE_IMAGE')
            layout.operator("custom.refresh_environment", text="2.刷新环境变量",icon='FILE_REFRESH')
            layout.separator()
            layout.operator("custom.remove_splash_system", text="移除启动画面",icon='TRASH')
        else:
            row = layout.row()
            row.enabled = False
            row.operator("custom.set_splash_system", text="1.设置启动画面", icon='FILE_IMAGE')
            
            row = layout.row()
            row.enabled = False
            row.operator("custom.refresh_environment", text="2.刷新环境变量", icon='FILE_REFRESH')
            layout.separator()
            row = layout.row()
            row.enabled = False
            row.operator("custom.remove_splash_system", text="移除启动画面", icon='TRASH')


def draw_help_menu(self, context):
    layout = self.layout
    layout.separator()
    layout.menu("CUSTOM_MT_splash_submenu", icon='IMAGE_BACKGROUND')
    

classes = (
    CUSTOM_OT_set_splash_system,
    CUSTOM_OT_remove_splash_system,
    CUSTOM_OT_run_as_admin,
    CUSTOM_OT_refresh_environment,
    CUSTOM_OT_show_current_splash,
    CUSTOM_MT_splash_submenu,
)

def register():
    for c in classes:
        bpy.utils.register_class(c)
    bpy.types.TOPBAR_MT_help.append(draw_help_menu)

def unregister():
    bpy.types.TOPBAR_MT_help.remove(draw_help_menu)
    for c in reversed(classes):
        bpy.utils.unregister_class(c)

if __name__ == "__main__":
    register()